#region References

using System;
using System.Diagnostics;
using System.Xml;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.OpenLibrary
{
	#region Header

		//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		//<Developers>
		//	<Developer>Brian Tomlin</Developer>
		//</Developers>
		//<SiteName>Hines OIFO</SiteName>
		//<CreationDate>10/19/2004</CreationDate>
		//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		//<summary>This class is used to create an appropriate HL7ProtocolMessage instance based on the message type in the MSH segment.</summary>

		#endregion

	/// <summary>
	/// Class HL7ProtocolMessageFactory
	/// </summary>
	public class HL7ProtocolMessageFactory
	{
		private const string CPRS_CLINICAL_ORDER_MESSAGE = "OMG^O19";
		private const string RESPONSE_TO_CPRS_CLINICAL_ORDER_MESSAGE = "ORG^O20";

		// CR 2962 BEGIN
		private const string VBECS_DISPENSE_STATUS_MESSAGE = "BPS~O29";
		private const string RESPONSE_TO_VBECS_DISPENSE_STATUS_MESSAGE = "BRP~O30";
		private const string BCE_TRANSFUSION_MESSAGE = "BTS~O31";
		private const string RESPONSE_TO_BCE_TRANSFUSION_MESSAGE = "BRT~O32";
		// CR 2962 END

		///<Developers>
		///	<Developer>EDS EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5802"> 
		///		<ExpectedInput>Valid HL7 Clinical Order Message string(OMG^O19)</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7ProtocolMessage.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5803"> 
		///		<ExpectedInput>Null hl7String input parameter</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5804"> 
		///		<ExpectedInput>Valid HL7 Response to Clinical Order Message string (ORG^O20)</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7ProtocolMessage</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5805"> 
		///		<ExpectedInput>Unsupported HL7 message type of (ADT^A08)</ExpectedInput>
		///		<ExpectedOutput>Null HL7ProtocolMessage</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates appropriate HL7ProtocolMessage instance based on message type
		/// </summary>
		/// <param name="hl7String"></param>
		/// <returns></returns>
		public HL7ProtocolMessage ParseCreateHL7ProtocolMessage( string hl7String )
		{
			HL7ProtocolMessage returnMessage = null;
			//
			if( hl7String == null )
			{
				throw( new ArgumentNullException( "hl7String" ) );
			}
			string messageType = HL7Utility.ParseGetHL7MessageTypeAndTriggerEventString( hl7String ).Trim();
			//
			switch( messageType )
			{
				case CPRS_CLINICAL_ORDER_MESSAGE:
				{
					returnMessage = new HL7OmgMessage( hl7String );
					break;
				}
				case RESPONSE_TO_CPRS_CLINICAL_ORDER_MESSAGE:
				{
					returnMessage = new HL7OrgMessage( hl7String );	
					break;
				}
				case VBECS_DISPENSE_STATUS_MESSAGE:
				{
					// CR 2962
					returnMessage = new HL7BpsMessage( hl7String );	
					break;
				}
				case RESPONSE_TO_VBECS_DISPENSE_STATUS_MESSAGE:
				{
					// CR 2962
					returnMessage = new HL7BrpMessage( hl7String );	
					break;
				}
				case BCE_TRANSFUSION_MESSAGE:
				{
					// CR 2962
					returnMessage = new HL7BtsMessage( hl7String );	
					break;
				}
				case RESPONSE_TO_BCE_TRANSFUSION_MESSAGE:
				{
					// CR 2962
					returnMessage = new HL7BrtMessage( hl7String );	
					break;
				}
			}
			//
			return returnMessage;
		}
	}
}